#!/bin/bash

#
# Codex Vault install script
#

if [[ $EUID -ne 0 ]]; then
   echo "This script must be run as root" 1>&2
   exit 1
fi

#-------------------------------------------------
# Logging
#-------------------------------------------------

logFile=/var/log/codex_install.log
cacheDirectory=/var/cache/codex

installDirectory="$(dirname "${0}")"
#scriptname="$(basename "${0}")"

#echo -n > ${logFile} # Disabled log file clearing
{ echo;
  echo "----------------------------------------------------------------------";
  date;
  echo "----------------------------------------------------------------------"; } >> ${logFile}

echoAndLog() {
    echo "${1}" 2>&1 | tee -a ${logFile}
}

#-------------------------------------------------
# Command line options
#-------------------------------------------------

uninstall=1
install=1
updateAVRs=1
stepFailed=0
changeSerialNum=0

while [[ ${#} -gt 0 ]]; do

    case "${1}" in
        "-u" | "--uninstall" )
            install=0
            updateAVRs=0
            shift 1
            ;;

        "-sf" | "--skipFirmwareUpdate" )
            updateAVRs=0
            shift 1
            ;;

        "-c" | "--changeSerial" )
            changeSerialNum=1
            shift 1
            ;;

        * )
            echoAndLog "Unrecognised argument \"${1}\""
            exit 1
            ;;
    esac
done

#-------------------------------------------------
# Hardware identification and Serial Number
#-------------------------------------------------

if [ ! -s /etc/codex/serial ] ; then
    echoAndLog "Can't find Vault serial number"
    changeSerialNum=1
fi

if [ ${changeSerialNum} -ne 0 ] ; then
    echoAndLog "Please enter full 8 digit serial number:"
    read -r NEWSERIAL

    if [ ${#NEWSERIAL} != 8 ] ; then
        echoAndLog "Serial number should be 8 digits long"
        exit 1
    fi

    if ! [[ ${NEWSERIAL} =~ ^[0-9]+$ ]] ; then
        echoAndLog "Serial number should be entirely numeric"
        exit 1
    fi

    echoAndLog "Setting serial to ${NEWSERIAL}"
    if ! mkdir -p /etc/codex ; then
        echoAndLog "Cannot create /etc/codex dir"
        exit 1
    fi

    if ! echo "${NEWSERIAL}" > /etc/codex/serial ; then
        echoAndLog "Cannot set serial number"
        exit 1
    fi
fi

if "${installDirectory}/isVaultXL" ; then

    # This is a Vault XL
    yumGroupName="Codex Vault XL Series"

elif "${installDirectory}/isVaultS" ; then

    # This is a Vault S
    yumGroupName="Codex Vault S Series"

else
    # Is this even a Vault at all?

    echoAndLog "Unable to identify Vault type - This does not appear to be a Codex Vault device"
    echoAndLog "Your serial number may be set incorrectly, please use \"install.sh -c\" to reenter it"
    exit 1

fi

#-------------------------------------------------
# Yum install
#-------------------------------------------------

echoAndLog

#-----------------------------
# Remove old installation

if [ ${uninstall} -ne 0 ] && [ $(($(yum list installed | grep -c "drserver"))) -ne 0 ] ; then

    if [ ${install} -ne 0 ] ; then
        echoAndLog "-------------------- Removing previous installation --------------------"
    else
        echoAndLog "------------------------ Removing installation -------------------------"
    fi

    yum -y --disablerepo "*" --enablerepo "codex" --setopt=clean_requirements_on_remove=1 group remove "${yumGroupName}" 2>&1 | tee -a ${logFile}

    # yum annoyingly sometimes returns 0 even on results that should be failures
    # the only way to be sure we succeeded is to look for a completion message in the log

    # "Complete!"" if it was uninstalled successfully

    if [ $(($(tail -n 1 ${logFile} | grep -c "Complete\\!"))) -ne 1 ] ; then
        stepFailed=1
        echoAndLog "Uninstall failed! Aborting..."
    fi
fi

#-----------------------------
# Install new software

if [ ${install} -ne 0 ] && [ ${stepFailed} -eq 0 ] ; then

    #-----------------------------
    # Transfer to cache if needed

    if [ "${installDirectory}" != "${cacheDirectory}" ] ; then

        tempCacheDirectory="${cacheDirectory}-temp"

        rm -rf ${tempCacheDirectory}
        mkdir -p ${tempCacheDirectory}

        cp -rf "${installDirectory}"/* ${tempCacheDirectory}

        # don't get rid of the old cache 'til the copy is complete

        rm -rf ${cacheDirectory}
        mv -f ${tempCacheDirectory} ${cacheDirectory} #in case the cache directory already exists

        cd ${cacheDirectory} || exit
    fi

    echoAndLog
    echoAndLog "----------------------- Installing new software ------------------------"
    echoAndLog

    ${cacheDirectory}/yumsetup.sh 2>&1 | tee -a ${logFile}

    yum -y --disablerepo "*" --enablerepo "codex" group install "${yumGroupName}" 2>&1 | tee -a ${logFile}

    if [ $(($(tail -n 1 ${logFile} | grep -c "Complete\\!"))) -ne 1 ] ; then
        stepFailed=1
        echoAndLog "Installation failed! Aborting..."
    fi
fi

#-------------------------------------------------
# AVR upgrade
#-------------------------------------------------

if [ ${updateAVRs} -ne 0 ] && [ ${stepFailed} -eq 0 ] ; then

    updateAVRFirmware --skipProcessStart 2>&1 | tee -a ${logFile} || stepFailed=1
fi

#-------------------------------------------------
#
#-------------------------------------------------

if [ ${stepFailed} -eq 0 ] ; then
    echoAndLog "Installer will now shut down the machine. Please restart to use your new Vault software."
    echo -n " Press 'Enter' to continue: "
    read -r
    shutdown now
fi
