<xsl:stylesheet
    version="1.0"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:exsl="http://exslt.org/common"
    extension-element-prefixes="exsl" >

    <!-- date conversion -->
    <xsl:template name="formatDate">
        <xsl:param name="dateString" />

        <xsl:choose>

            <xsl:when test="(string-length($dateString) = 8) and (number($dateString) != Nan)">

                <xsl:variable name="year" select="substring($dateString, 1, 4)" />
                <xsl:variable name="month" select="substring($dateString, 5, 2)" />
                <xsl:variable name="day" select="substring($dateString, 7, 2)" />

                <xsl:value-of select="format-number($day, '##')" />
                <xsl:text> </xsl:text>

                <xsl:choose>
                    <xsl:when test="$month = '01'">Jan</xsl:when>
                    <xsl:when test="$month = '02'">Feb</xsl:when>
                    <xsl:when test="$month = '03'">Mar</xsl:when>
                    <xsl:when test="$month = '04'">Apr</xsl:when>
                    <xsl:when test="$month = '05'">May</xsl:when>
                    <xsl:when test="$month = '06'">Jun</xsl:when>
                    <xsl:when test="$month = '07'">Jul</xsl:when>
                    <xsl:when test="$month = '08'">Aug</xsl:when>
                    <xsl:when test="$month = '09'">Sep</xsl:when>
                    <xsl:when test="$month = '10'">Oct</xsl:when>
                    <xsl:when test="$month = '11'">Nov</xsl:when>
                    <xsl:when test="$month = '12'">Dec</xsl:when>
                </xsl:choose>

                <xsl:text> </xsl:text>
                <xsl:value-of select="format-number($year, '00')" />

            </xsl:when>

            <xsl:otherwise>
              <xsl:value-of select="$dateString"/>
            </xsl:otherwise>

        </xsl:choose>

    </xsl:template>

    <!-- convert from our enum for frame rates -->
    <xsl:template name="frameRateDecode">
        <xsl:param name="frameRateEnum" />

        <xsl:choose>
            <xsl:when test="$frameRateEnum = 0">23.98</xsl:when>
            <xsl:when test="$frameRateEnum = 1">24</xsl:when>
            <xsl:when test="$frameRateEnum = 2">25</xsl:when>
            <xsl:when test="$frameRateEnum = 3">29.97</xsl:when>
            <xsl:when test="$frameRateEnum = 4">30</xsl:when>
            <xsl:when test="$frameRateEnum = 5">59.94</xsl:when>
            <xsl:when test="$frameRateEnum = 6">60</xsl:when>
            <xsl:when test="$frameRateEnum = 7">47.95</xsl:when>
            <xsl:when test="$frameRateEnum = 8">48</xsl:when>
            <xsl:when test="$frameRateEnum = 9">50</xsl:when>
            <xsl:when test="$frameRateEnum = 11">95.90</xsl:when>
            <xsl:when test="$frameRateEnum = 12">96</xsl:when>
            <xsl:when test="$frameRateEnum = 13">100</xsl:when>
            <xsl:when test="$frameRateEnum = 14">119.88</xsl:when>
            <xsl:when test="$frameRateEnum = 15">120</xsl:when>
            <xsl:otherwise>???</xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- recursive function to count up total duration of shots -->
    <xsl:template name="executiveSummary">
        <xsl:param name="currSumSecs" />
        <xsl:param name="currSumSize" />
        <xsl:param name="count" />
        <xsl:param name="runningProd" />
        <xsl:param name="runningDate" />
        <xsl:param name="anyExpIndex" />
        <xsl:param name="anyWhiteBal" />
        <xsl:param name="anyTint" />

        <xsl:variable name="numShotFrames" select="CodexShotXMLList/CodexShotMetadata[number($count)]/Timecode/Channel/Samples" />
        <xsl:variable name="ourFrameRateEnum" select="CodexShotXMLList/CodexShotMetadata[number($count)]/Video/VideoFormat/FrameRate" />
        <xsl:variable name="frameRate">
            <xsl:call-template name="frameRateDecode">
                <xsl:with-param name="frameRateEnum">
                    <xsl:value-of select="$ourFrameRateEnum"/>
                </xsl:with-param>
            </xsl:call-template>
        </xsl:variable>

        <xsl:variable name="shotDuration" select="$numShotFrames div $frameRate" />
        <xsl:variable name="sumSecs" select="number($currSumSecs) + number($shotDuration)" />

        <xsl:variable name="shotSize" select="CodexShotXMLList/CodexShotMetadata[number($count)]/SizeInGB" />
        <xsl:variable name="sumSize" select="number($currSumSize) + number($shotSize)" />

        <xsl:variable name="curProd">
            <xsl:choose>
                <xsl:when test="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ProductionName != ''">
                    <xsl:value-of select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ProductionName" />
                </xsl:when>
            <xsl:otherwise>&lt;Not Set&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="prod">
            <xsl:choose>
                <xsl:when test="$runningProd = 0 or $runningProd = $curProd">
                    <xsl:value-of select="$curProd" />
                </xsl:when>
                <xsl:otherwise>&lt;mixed&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curDate">
            <xsl:choose>
                <xsl:when test="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ShootDate != ''">
                    <xsl:value-of select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ShootDate" />
                </xsl:when>
                <xsl:otherwise>&lt;Not Set&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="date">
            <xsl:choose>
                <xsl:when test="$runningDate = 0 or $runningDate = $curDate">
                    <xsl:value-of select="$curDate" />
                </xsl:when>
                <xsl:otherwise>&lt;mixed&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curExpIndex" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ExposureIndex" />
        <xsl:variable name="expIndexFlag">
            <xsl:choose>
                <xsl:when test="$anyExpIndex = 1 or $curExpIndex != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curWhiteBal" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/WhiteBalance" />
        <xsl:variable name="whiteBalFlag">
            <xsl:choose>
                <xsl:when test="$anyWhiteBal = 1 or $curWhiteBal != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curTint" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/GreenTint" />
        <xsl:variable name="tintFlag">
            <xsl:choose>
                <xsl:when test="$anyTint = 1 or $curTint != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:choose>
            <xsl:when test="number($count) &gt; 1">
                <xsl:call-template name="executiveSummary">
                    <xsl:with-param name="currSumSecs"><xsl:value-of select="$sumSecs" /></xsl:with-param>
                    <xsl:with-param name="currSumSize"><xsl:value-of select="$sumSize" /></xsl:with-param>
                    <xsl:with-param name="count"><xsl:value-of select="number($count - 1)" /></xsl:with-param>
                    <xsl:with-param name="runningProd"><xsl:value-of select="$prod" /></xsl:with-param>
                    <xsl:with-param name="runningDate"><xsl:value-of select="$date" /></xsl:with-param>
                    <xsl:with-param name="anyExpIndex"><xsl:value-of select="$expIndexFlag" /></xsl:with-param>
                    <xsl:with-param name="anyWhiteBal"><xsl:value-of select="$whiteBalFlag" /></xsl:with-param>
                    <xsl:with-param name="anyTint"><xsl:value-of select="$tintFlag" /></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <results>
                    <duration><xsl:value-of select="$sumSecs" /></duration>
                    <size><xsl:value-of select="$sumSize" /></size>
                    <productionName><xsl:value-of select="$prod" /></productionName>
                    <shootDate><xsl:value-of select="$date" /></shootDate>
                    <exposureIndexSet><xsl:value-of select="$expIndexFlag" /></exposureIndexSet>
                    <whiteBalanceSet><xsl:value-of select="$whiteBalFlag" /></whiteBalanceSet>
                    <greenTintSet><xsl:value-of select="$tintFlag" /></greenTintSet>
                </results>
            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>

    <!-- the main gubbins -->
    <xsl:template match="/">

        <html>

            <head>
              <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
              <link rel="stylesheet" type="text/css" href="report.css" />
              <title>Codex Shot Report</title>
            </head>

            <body>

                <!-- set-up some (possibly) useful variables -->

                <xsl:variable name="numShots" select="count(CodexShotXMLList/CodexShotMetadata)" />
                <xsl:variable name="totalFrames" select="sum(CodexShotXMLList/CodexShotMetadata/Video/Channel/Samples)" />

                <xsl:variable name="temp">
                    <xsl:call-template name="executiveSummary">
                        <xsl:with-param name="currSumSecs">0</xsl:with-param>
                        <xsl:with-param name="currSumSize">0</xsl:with-param>
                        <xsl:with-param name="count"><xsl:value-of select="$numShots" /></xsl:with-param>
                        <xsl:with-param name="runningProd">0</xsl:with-param>
                        <xsl:with-param name="runningDate">0</xsl:with-param>
                        <xsl:with-param name="anyExpIndex">0</xsl:with-param>
                        <xsl:with-param name="anyWhiteBal">0</xsl:with-param>
                        <xsl:with-param name="anyTint">0</xsl:with-param>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:variable name="totalDurationSecs" select="exsl:node-set($temp)/results/duration" />
                <xsl:variable name="totalSizeGB" select="exsl:node-set($temp)/results/size" />
                <xsl:variable name="productionName" select="exsl:node-set($temp)/results/productionName" />
                <xsl:variable name="shootDate" select="exsl:node-set($temp)/results/shootDate" />
                <xsl:variable name="showExpIndex" select="exsl:node-set($temp)/results/exposureIndexSet" />
                <xsl:variable name="showWhiteBal" select="exsl:node-set($temp)/results/whiteBalanceSet" />
                <xsl:variable name="showGreenTint" select="exsl:node-set($temp)/results/greenTintSet" />

                <xsl:variable name="hours" select="floor($totalDurationSecs div (60*60))" />
                <xsl:variable name="mins" select="floor(($totalDurationSecs div 60) - ($hours*60))" />
                <xsl:variable name="secs" select="ceiling($totalDurationSecs) mod 60" />

                <xsl:variable name="author" select="CodexShotXMLList/Author" />
                <xsl:variable name="overrideEnum" select="CodexShotXMLList/FrameRateOverride" />

                <xsl:variable name="overrideFps">
                    <xsl:call-template name="frameRateDecode">
                        <xsl:with-param name="frameRateEnum"><xsl:value-of select="$overrideEnum"/></xsl:with-param>
                    </xsl:call-template>
                </xsl:variable>

                <!-- Print logo, header and column titles -->
                <table width="1000" height="98" border="0" cellpadding="0" cellspacing="0">

                    <!-- logo -->
                    <tr>
                        <td colspan="18"><div align="right"><img src="report-header.gif" alt="" width="709" height="111" /></div></td>
                    </tr>
                    <tr>
                        <td colspan="18">
                            <img src="report-transp.gif" alt="" width="14" height="10" />
                        </td>
                    </tr>

                    <!-- production/author/shootdate -->
                    <tr>
                        <td colspan="18"><table width="100%" border="0" cellpadding="0" cellspacing="0" bgcolor="#CCCCCC">
                            <tr>
                                <td width="14" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="14" height="12" />
                                </td>

                                <td width="292" valign="middle" bgcolor="#dfdfdf"><div align="left">
                                    <strong>PRODUCTION: </strong>
                                    <xsl:value-of select="$productionName" />
                                </div></td>

                                <td width="70" valign="middle" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="14" height="12" />
                                </td>

                                <td width="292" valign="middle" bgcolor="#dfdfdf"><div align="left">
                                  <xsl:if test="$author != ''">
                                    <strong>AUTHOR: </strong>
                                    <xsl:value-of select="$author" />
                                    <xsl:text>; </xsl:text>
                                  </xsl:if>
                                  <xsl:value-of select="CodexShotXMLList/ReportDate" />
                                </div></td>

                                <td width="45" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="100" height="20" />
                                </td>

                                <td width="207" valign="middle" bgcolor="#dfdfdf"><div align="right"><strong>SHOOT DATE: </strong>
                                  <xsl:variable name="shootDateTidy">
                                    <xsl:call-template name="formatDate">
                                      <xsl:with-param name="dateString">
                                        <xsl:value-of select="$shootDate"/>
                                      </xsl:with-param>
                                    </xsl:call-template>
                                  </xsl:variable>
                                  <xsl:value-of select="$shootDateTidy" />
                                </div></td>

                                <td width="18" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="18" height="20" />
                                </td>
                            </tr>
                        </table></td>
                    </tr>

                    <!-- summary line -->
                    <tr>
                        <td colspan="18"><img src="report-transp.gif" alt="" width="14" height="14" /></td>
                    </tr>

                    <tr>
                        <td colspan="18">
                            <table width="100%" border="0" cellpadding="0" cellspacing="0" bgcolor="#CCCCCC">
                                <tr>
                                    <td width="14" height="20" bgcolor="#dfdfdf">
                                        <img src="report-transp.gif" alt="" width="14" height="20" />
                                    </td>

                                    <td valign="middle" bgcolor="#dfdfdf"><div align="left"><strong>SUMMARY: </strong>

                                        <xsl:value-of select="$numShots" />
                                        <xsl:choose>
                                            <xsl:when test="number($numShots) > 1">
                                                <xsl:text> clips; </xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text> clip; </xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>

                                        <xsl:value-of select="format-number($totalFrames, ',###,###')" />
                                        <xsl:choose>
                                            <xsl:when test="number($totalFrames) > 1">
                                                <xsl:text> frames; </xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text> frame; </xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>

                                        <xsl:choose>
                                            <xsl:when test="string(number($totalSizeGB))='NaN'">0</xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="format-number($totalSizeGB, ',###,###')" />
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:text> GB; </xsl:text>

                                        <xsl:if test="number($hours) > 0">
                                            <xsl:value-of select="$hours" />
                                            <xsl:choose>
                                                <xsl:when test="number($hours)=1">
                                                    <xsl:text> hr </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> hrs </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test="number($mins) > 0">
                                            <xsl:value-of select="$mins" />
                                            <xsl:choose>
                                                <xsl:when test="number($mins)=1">
                                                    <xsl:text> min </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> mins </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test="number($secs) > 0">
                                            <xsl:value-of select="ceiling($secs)" />
                                            <xsl:choose>
                                                <xsl:when test="number($secs)=1">
                                                    <xsl:text> sec </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> secs </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test= "(string(number($secs))='NaN') and (string(number($hours))='NaN') and (string(number($secs))='NaN')">
                                            <xsl:text> 0 secs</xsl:text>
                                        </xsl:if>

                                        <xsl:if test="$overrideFps != '???'">
                                            <xsl:text> (project frame rate: </xsl:text>
                                            <xsl:value-of select="$overrideFps"/>
                                            <xsl:text>fps)</xsl:text>
                                        </xsl:if>
                                    </div></td>
                                </tr>
                            </table>
                        </td>
                    </tr>

                    <!-- gap between summary and table column headings-->
                    <tr>
                        <td height="15" colspan="18" align="left" valign="top">
                            <img src="report-transp.gif" alt="" width="14" height="14" />
                        </td>
                    </tr>

                    <!-- table column headings -->
                    <tr>

                        <td width="10" height="17" valign="middle" bgcolor="#dfdfdf"><h5>
                            <img src="report-transp.gif" alt="" width="11" height="20" />
                        </h5></td>

                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="left">ROLL
                        </h5></td>
                        <td height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="left">CLIP
                        </h5></td>
                        <td width="50" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                            <xsl:if test="$overrideFps != '???'">
                                <xsl:text>ORIGINAL</xsl:text>
                                <br/>
                            </xsl:if>
                            <xsl:text>FPS</xsl:text>
                        </h5></td>

                        <xsl:choose>
                            <xsl:when test="$overrideFps != '???'">
                                <td width="70"  height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">NATIVE TC
                                </h5></td>
                                <td width="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                                    <xsl:text>PROJECT</xsl:text>
                                    <br/>
                                    <xsl:text>START TC</xsl:text>
                                </h5></td>
                                <td width="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                                    <xsl:text>PROJECT</xsl:text>
                                    <br/>
                                    <xsl:text>END TC</xsl:text>
                                </h5></td>
                            </xsl:when>
                            <xsl:otherwise>
                                <td widht="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">START TC
                                </h5></td>
                                <td width="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">END TC
                                </h5></td>
                            </xsl:otherwise>
                        </xsl:choose>

                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">FRAMES
                        </h5></td>
                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">CHANS
                        </h5></td>
                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">DIMENSIONS
                        </h5></td>
                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">FORMAT
                        </h5></td>
                        <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">GB
                        </h5></td>

                        <xsl:if test="$showExpIndex = 1">
                          <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                              <xsl:text>EXP.</xsl:text>
                              <br/>
                              <xsl:text>INDEX</xsl:text>
                          </h5></td>
                        </xsl:if>

                        <xsl:if test="$showWhiteBal = 1">
                          <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                              <xsl:text>WHITE</xsl:text>
                              <br/>
                              <xsl:text>BAL</xsl:text>
                          </h5></td>
                        </xsl:if>

                        <xsl:if test="$showGreenTint = 1">
                          <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">TINT
                          </h5></td>
                        </xsl:if>

                        <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">
                            <xsl:text>COMMENTS</xsl:text>
                        </h5></td>

                        <td width="10" height="17" valign="middle" bgcolor="#dfdfdf"><h5>
                            <img src="report-transp.gif" alt="" width="18" height="17" />
                        </h5></td>
                    </tr>

                    <!-- loop round printing details for each shot -->
                    <xsl:for-each select="CodexShotXMLList/CodexShotMetadata">

                        <!-- set sorting up-->
                        <xsl:sort select="PropertyStrings/OriginalRoll" data-type="text" order="ascending" />
                        <xsl:sort select="Timecode/Channel/StartTc" data-type="text" order="ascending" />

                        <xsl:variable name="sortedIdx" select="position()"/>

                        <!-- create some per shot variables -->
                        <xsl:variable name="tcFrameRate" select="Timecode/Channel/TcFormat/Rate"/>
                        <xsl:variable name="tcFieldRate">
                          <xsl:choose>
                            <xsl:when test="Timecode/Channel/TcFormat/Field = 'true'">2</xsl:when>
                            <xsl:otherwise>1</xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>
                        <xsl:variable name="tcDropFlag" select="Timecode/Channel/TcFormat/DropFrame"/>
                        <xsl:variable name="frameRate">
                              <xsl:call-template name="frameRateDecode">
                                  <xsl:with-param name="frameRateEnum">
                                      <xsl:value-of select="Video/VideoFormat/FrameRate"/>
                                  </xsl:with-param>
                              </xsl:call-template>
                        </xsl:variable>

                        <xsl:variable name="pixelFormat">
                          <xsl:choose>
                            <xsl:when test="Video/VideoFormat/PixFormat = 0">8bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 1">16bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 2">10bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 3">10bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 4">32bit float</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 5">16bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 6">32bit float</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 7">12bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 8">12bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 9">12bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 10">12bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 11">12bit</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 12">16bit float</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 13">16bit float</xsl:when>
                            <xsl:when test="Video/VideoFormat/PixFormat = 14">10bit</xsl:when>
                            <xsl:otherwise>???</xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>

                        <xsl:variable name="sampleFormat">
                          <xsl:choose>
                            <xsl:when test="Video/VideoFormat/SampFormat = 0">4:4:4</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 1">4:4:4</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 2">4:2:2</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 3">4:2:2</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 4">4:4:4:4</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 5">4:4:4:4</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 6">Bayer</xsl:when>
                            <xsl:when test="Video/VideoFormat/SampFormat = 7">1</xsl:when>
                            <xsl:otherwise>???</xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>

                        <xsl:variable name="colourSpace">
                          <xsl:choose>
                            <xsl:when test="Video/VideoFormat/ColSpace = 0">YCbCr</xsl:when>
                            <xsl:when test="Video/VideoFormat/ColSpace = 1">RGB</xsl:when>
                            <xsl:when test="Video/VideoFormat/ColSpace = 2">CMYK</xsl:when>
                            <xsl:when test="Video/VideoFormat/ColSpace = 3">LUMA</xsl:when>
                            <xsl:otherwise>???</xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>

                        <xsl:variable name="compressionFactor" select="Video/VideoFormat/CompressionFactor"/>
                        <xsl:variable name="compressionType" select="Video/VideoFormat/CompressionType"/>

                        <xsl:variable name="numVidChans" select="Video/@numChans"/>
                        <xsl:variable name="numAudChans" select="Audio/@numChans"/>

                        <xsl:variable name="numFrames" select="Video/Channel/Samples"/>

                        <xsl:variable name="startTc" select="Timecode/Channel/StartTc"/>
                        <xsl:variable name="tcStartH" select="substring($startTc, 1, 2)"/>
                        <xsl:variable name="tcStartM" select="substring($startTc, 4, 2)"/>
                        <xsl:variable name="tcStartS" select="substring($startTc, 7, 2)"/>
                        <xsl:variable name="tcStartF" select="substring($startTc, 10, 2)"/>

                        <xsl:variable name="projectFps">
                          <xsl:choose>
                            <xsl:when test="$overrideFps != '???'">
                              <xsl:value-of select="$overrideFps" />
                            </xsl:when>
                            <xsl:otherwise><xsl:value-of select="$frameRate" /></xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>

                        <xsl:variable name="startFrames" select="((((($tcStartH*60)+$tcStartM)*60)+$tcStartS)*ceiling($frameRate))+$tcStartF"/>
                        <xsl:variable name="projTcStartH" select="floor($startFrames div (60*60*ceiling($projectFps)))"/>
                        <xsl:variable name="projTcStartM" select="floor($startFrames div (60*ceiling($projectFps))) - floor($projTcStartH * 60)" />
                        <xsl:variable name="projTcStartS" select="floor($startFrames div ceiling($projectFps)) mod 60" />
                        <xsl:variable name="projTcStartF" select="$startFrames mod ceiling($projectFps)" />

                        <xsl:variable name="endFrames" select="$startFrames + $numFrames - 1" />
                        <xsl:variable name="projTcEndH" select="floor($endFrames div (60*60*ceiling($projectFps)))" />
                        <xsl:variable name="projTcEndM" select="floor($endFrames div (60*ceiling($projectFps))) - floor($projTcEndH * 60)"/>
                        <xsl:variable name="projTcEndS" select="floor($endFrames div ceiling($projectFps)) mod 60" />
                        <xsl:variable name="projTcEndF" select="$endFrames mod ceiling($projectFps)" />

                        <xsl:variable name="smallcase" select="'abcdefghijklmnopqrstuvwxyz'" />
                        <xsl:variable name="uppercase" select="'ABCDEFGHIJKLMNOPQRSTUVWXYZ'" />

                        <!-- print the table row for this shot -->
                        <tr height="12">

                            <!-- set background colour for alternating stripes -->
                            <xsl:if test="(number($sortedIdx) mod 2) = 0">
                                <xsl:attribute name="bgcolor">#dfdfdf</xsl:attribute>
                            </xsl:if>

                            <td><h6>
                                <xsl:text>&#160;</xsl:text>
                            </h6></td>

                            <td><h6 align="left">
                                <xsl:value-of select="PropertyStrings/OriginalRoll" />
                            </h6></td>

                            <td><h6 align="left">
                                <xsl:value-of select="PropertyStrings/ShotName" />
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:value-of select="$frameRate" />
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:choose>
                                    <xsl:when test="$startTc != ''">
                                        <xsl:value-of select="$startTc" />
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:text>??:??:??:??</xsl:text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h6></td>

                            <xsl:if test="$overrideFps != '???'">
                                <td>
                                    <h6 align="center">
                                        <xsl:choose>
                                            <xsl:when test="$startTc != ''">
                                                <xsl:value-of select="format-number($projTcStartH, '00')"/>
                                                <xsl:text>:</xsl:text>
                                                <xsl:value-of select="format-number($projTcStartM, '00')"/>
                                                <xsl:text>:</xsl:text>
                                                <xsl:value-of select="format-number($projTcStartS, '00')"/>
                                                <xsl:text>:</xsl:text>
                                                <xsl:value-of select="format-number($projTcStartF, '00')"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text>??:??:??:??</xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </h6>
                                </td>
                            </xsl:if>

                            <td><h6 align="center">
                                <xsl:choose>
                                    <xsl:when test="$startTc != ''">
                                        <xsl:variable name="shotTcRate" select="$tcFrameRate * $tcFieldRate" />
                                        <xsl:variable name="tcStartFrames" select="((((($tcStartH*60)+$tcStartM)*60)+$tcStartS)*ceiling($shotTcRate))+$tcStartF"/>
                                        <xsl:variable name="tcEndFrames" select="$tcStartFrames + $numFrames - 1" />
                                        <xsl:variable name="shotTcEndH" select="floor($tcEndFrames div (60*60*ceiling($shotTcRate)))" />
                                        <xsl:variable name="shotTcEndM" select="floor($tcEndFrames div (60*ceiling($shotTcRate))) - floor($shotTcEndH * 60)"/>
                                        <xsl:variable name="shotTcEndS" select="floor($tcEndFrames div ceiling($shotTcRate)) mod 60" />
                                        <xsl:variable name="shotTcEndF" select="$tcEndFrames mod ceiling($shotTcRate)" />

                                        <xsl:value-of select="format-number($shotTcEndH, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($shotTcEndM, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($shotTcEndS, '00')"/>
                                        <xsl:choose>
                                            <xsl:when test="$tcDropFlag = 'true'">
                                                <xsl:text>;</xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text>:</xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:value-of select="format-number($shotTcEndF, '00')"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:text>??:??:??:??</xsl:text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:value-of select="$numFrames" />
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:text>V</xsl:text>
                                <xsl:value-of select="Video/@numChans" />
                                <xsl:if test="Audio/@numChans">
                                    <xsl:text>A</xsl:text>
                                    <xsl:value-of select="Audio/@numChans" />
                                </xsl:if>
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:value-of select="CorrectedWidth" />
                                <xsl:text>x</xsl:text>
                                <xsl:value-of select="CorrectedHeight" />
                            </h6></td>

                            <td><h6 align="center">
                                <!-- If the shot is a RED shot the just display its REDCode -->
                                <xsl:choose>
                                    <xsl:when test="$compressionType = 8">
                                        <xsl:variable name="redCode" select="PropertyStrings/RED.Redcode" />
                                        <xsl:value-of select="$redCode" />
                                    </xsl:when>
                                    <!-- Otherwise displays its full pixel format -->
                                    <xsl:otherwise>
                                        <xsl:if test="$compressionType = 12">
                                            <xsl:text>H264 </xsl:text>
                                        </xsl:if>
                                        <xsl:value-of select="$pixelFormat" />
                                        <xsl:text> </xsl:text>
                                        <xsl:choose>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and starts-with( translate(PropertyStrings/SourceDevice, $smallcase, $uppercase), 'ARRI ALEXA')">ARRIRAW</xsl:when>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and starts-with( translate(PropertyStrings/SourceDevice, $smallcase, $uppercase), 'ALEXA')">ARRIRAW</xsl:when>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and PropertyStrings/SourceDevice = 'ActionCAM'">Action Cam RAW</xsl:when>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and PropertyStrings/SourceDevice = 'ActionCAMMono'">Action Cam RAW</xsl:when>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and PropertyStrings/SourceDevice = 'Varicam35'">VRAW</xsl:when>
                                            <xsl:when test="Video/VideoFormat/SampFormat = 6 and starts-with( translate(PropertyStrings/SourceDevice, $smallcase, $uppercase), 'EOS')">Canon RAW</xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$sampleFormat" />
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:if test="Video/VideoFormat/SampFormat != 6">
                                                <xsl:text> </xsl:text>
                                                <xsl:value-of select="$colourSpace" />
                                        </xsl:if>
                                        <xsl:if test="$compressionType != 0 and $compressionType != 12">
                                            <xsl:text> </xsl:text>
                                            <xsl:choose>
                                                <xsl:when test="$compressionFactor = 4000">
                                                    <xsl:text>SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 17">
                                                    <xsl:text>F65 RAW HQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 33">
                                                    <xsl:text>F65 RAW SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 49">
                                                    <xsl:text>F65 RAW LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 65">
                                                    <xsl:text>F65 RAW NC</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 802">
                                                    <xsl:text>F65 RAW SQ HFR</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 1057">
                                                    <xsl:text>F55 RAW SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 1073">
                                                    <xsl:text>F55 RAW LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 3873">
                                                    <xsl:text>F5 RAW SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 3889">
                                                    <xsl:text>F5 RAW LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 7969 or $compressionFactor = 12065">
                                                    <xsl:text>CV RAW SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 7985 or $compressionFactor = 12081">
                                                    <xsl:text>CV RAW LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 14101">
                                                    <xsl:text>SStP SR-SQ</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 14100">
                                                    <xsl:text>SStP SR-LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 14099">
                                                    <xsl:text>SStP</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18195">
                                                    <xsl:text>XAVC</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18196">
                                                    <xsl:text>F55 XOCN LT</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18197">
                                                    <xsl:text>F55 XOCN ST</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18198">
                                                    <xsl:text>F5 XOCN LT</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18199">
                                                    <xsl:text>F5 XOCN ST</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18201 or $compressionFactor = 18203 or $compressionFactor = 18205">
                                                    <xsl:text>VENICE XOCN ST</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionFactor = 18200 or $compressionFactor = 18202 or $compressionFactor = 18204">
                                                    <xsl:text>VENICE XOCN LITE</xsl:text>
                                                </xsl:when>
                                                <xsl:when test="$compressionType = 15">
                                                    <xsl:text>HDE</xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="$compressionFactor div 1024" />
                                                    <xsl:text>:1</xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:value-of select="SizeInGB" />
                            </h6></td>

                            <xsl:if test="$showExpIndex = 1">
                                <td><h6 align="center">
                                    <xsl:value-of select="PropertyStrings/ExposureIndex" />
                                </h6></td>
                            </xsl:if>

                            <xsl:if test="$showWhiteBal = 1">
                                <td><h6 align="center">
                                    <xsl:value-of select="PropertyStrings/WhiteBalance" />
                                </h6></td>
                            </xsl:if>

                            <xsl:if test="$showGreenTint = 1">
                                <td><h6 align="center">
                                    <xsl:value-of select="PropertyStrings/GreenTint" />
                                </h6></td>
                            </xsl:if>

                            <td><h6 align="center">
                                <xsl:value-of select="PropertyStrings/Comments" />
                            </h6></td>

                            <td><h6>
                                <xsl:text>&#160;</xsl:text>
                            </h6></td>
                        </tr>
                    </xsl:for-each>
                </table>
            </body>
        </html>
    </xsl:template>
</xsl:stylesheet>
