<xsl:stylesheet
    version="1.0"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:exsl="http://exslt.org/common"
    extension-element-prefixes="exsl" >

    <!-- date conversion -->
    <xsl:template name="formatDate">
        <xsl:param name="dateString" />

        <xsl:choose>

            <xsl:when test="(string-length($dateString) = 8) and (number($dateString) != Nan)">

                <xsl:variable name="year" select="substring($dateString, 1, 4)" />
                <xsl:variable name="month" select="substring($dateString, 5, 2)" />
                <xsl:variable name="day" select="substring($dateString, 7, 2)" />

                <xsl:value-of select="format-number($day, '##')" />
                <xsl:text> </xsl:text>

                <xsl:choose>
                    <xsl:when test="$month = '01'">Jan</xsl:when>
                    <xsl:when test="$month = '02'">Feb</xsl:when>
                    <xsl:when test="$month = '03'">Mar</xsl:when>
                    <xsl:when test="$month = '04'">Apr</xsl:when>
                    <xsl:when test="$month = '05'">May</xsl:when>
                    <xsl:when test="$month = '06'">Jun</xsl:when>
                    <xsl:when test="$month = '07'">Jul</xsl:when>
                    <xsl:when test="$month = '08'">Aug</xsl:when>
                    <xsl:when test="$month = '09'">Sep</xsl:when>
                    <xsl:when test="$month = '10'">Oct</xsl:when>
                    <xsl:when test="$month = '11'">Nov</xsl:when>
                    <xsl:when test="$month = '12'">Dec</xsl:when>
                </xsl:choose>

                <xsl:text> </xsl:text>
                <xsl:value-of select="format-number($year, '00')" />

            </xsl:when>

            <xsl:otherwise>
              <xsl:value-of select="$dateString"/>
            </xsl:otherwise>

        </xsl:choose>

    </xsl:template>

    <!-- convert from our enum for frame rates -->
    <xsl:template name="frameRateDecode">
        <xsl:param name="frameRateEnum" />

        <xsl:choose>
            <xsl:when test="$frameRateEnum = 0">23.98</xsl:when>
            <xsl:when test="$frameRateEnum = 1">24</xsl:when>
            <xsl:when test="$frameRateEnum = 2">25</xsl:when>
            <xsl:when test="$frameRateEnum = 3">29.97</xsl:when>
            <xsl:when test="$frameRateEnum = 4">30</xsl:when>
            <xsl:when test="$frameRateEnum = 5">59.94</xsl:when>
            <xsl:when test="$frameRateEnum = 6">60</xsl:when>
            <xsl:when test="$frameRateEnum = 7">47.95</xsl:when>
            <xsl:when test="$frameRateEnum = 8">48</xsl:when>
            <xsl:when test="$frameRateEnum = 9">50</xsl:when>
            <xsl:when test="$frameRateEnum = 11">95.90</xsl:when>
            <xsl:when test="$frameRateEnum = 12">96</xsl:when>
            <xsl:when test="$frameRateEnum = 13">100</xsl:when>
            <xsl:when test="$frameRateEnum = 14">119.88</xsl:when>
            <xsl:when test="$frameRateEnum = 15">120</xsl:when>
            <xsl:otherwise>???</xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getRegionXOffset">
        <xsl:param name="qcNode" />
        <xsl:choose>
            <xsl:when test="not(Region)">0</xsl:when>
            <xsl:otherwise>
                <xsl:variable name="regionString" select="Region" />
                <xsl:variable name="positionString" select="substring-after($regionString, '@')" />
                <xsl:variable name="xOffsetString" select="substring-before($positionString, ',')" />
                <xsl:value-of select="$xOffsetString" />
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getRegionYOffset">
        <xsl:param name="qcNode" />
        <xsl:choose>
            <xsl:when test="not(Region)">0</xsl:when>
            <xsl:otherwise>
                <xsl:variable name="regionString" select="Region" />
                <xsl:variable name="positionString" select="substring-after($regionString, '@')" />
                <xsl:variable name="yOffsetString" select="substring-after($positionString, ',')" />
                <xsl:value-of select="$yOffsetString" />
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getRegionWidth">
        <xsl:param name="qcNode" />
        <xsl:choose>
            <xsl:when test="not(Region)">
                <xsl:variable name="canvasString" select="Canvas" />
                <xsl:variable name="widthString" select="substring-before($canvasString, 'x')" />
                <xsl:value-of select="$widthString" />
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="regionString" select="Region" />
                <xsl:variable name="sizeString" select="substring-before($regionString, '@')" />
                <xsl:variable name="widthString" select="substring-before($sizeString, 'x')" />
                <xsl:value-of select="$widthString" />
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getRegionHeight">
        <xsl:param name="qcNode" />
        <xsl:choose>
            <xsl:when test="not(Region)">
                <xsl:variable name="canvasString" select="Canvas" />
                <xsl:variable name="heightString" select="substring-after($canvasString, 'x')" />
                <xsl:value-of select="$heightString" />
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="regionString" select="Region" />
                <xsl:variable name="sizeString" select="substring-before($regionString, '@')" />
                <xsl:variable name="heightString" select="substring-after($sizeString, 'x')" />
                <xsl:value-of select="$heightString" />
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getFramesInQCMark">
        <xsl:param name="framesInClip" />
        <xsl:choose>
            <xsl:when test="not(Range)">
                <xsl:value-of select="$framesInClip" />
            </xsl:when>
            <xsl:when test="Range/@duration">
                <xsl:value-of select="Range/@duration" />
            </xsl:when>
            <xsl:otherwise>
                1
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getFirstFrameInQCMark">
        <xsl:choose>
            <xsl:when test="Range/@first">
                <xsl:value-of select="Range/@first" />
            </xsl:when>
            <xsl:otherwise>
                0
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getQCMarkType">
        <xsl:choose>
            <xsl:when test="not(Range)">
                Entire Clip
            </xsl:when>
            <xsl:when test="Range/@duration">
                Range
            </xsl:when>
            <xsl:otherwise>
                Frame
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <xsl:template name="getRegionLocation">
        <xsl:choose>
            <xsl:when test="Region">
                 <xsl:variable name="regionString" select="Region" />

                <xsl:variable name="positionString" select="substring-after($regionString, '@')" />
                <xsl:variable name="sizeString" select="substring-before($regionString, '@')" />

                <xsl:variable name="xOffsetString" select="substring-before($positionString, ',')" />
                <xsl:variable name="yOffsetString" select="substring-after($positionString, ',')" />

                <xsl:variable name="widthString" select="substring-before($sizeString, 'x')" />
                <xsl:variable name="heightString" select="substring-after($sizeString, 'x')" />

                <xsl:variable name="halfWidthString" select="number($widthString) div 2" />
                <xsl:variable name="halfHeightString" select="number($heightString) div 2" />

                <xsl:variable name="regionCentreX" select="number($xOffsetString) + number($halfWidthString)" />
                <xsl:variable name="regionCentreY" select="number($yOffsetString) + number($halfHeightString)" />

                <xsl:variable name="canvasString" select="Canvas" />
                <xsl:variable name="clipWidthString" select="substring-before($canvasString, 'x')" />
                <xsl:variable name="clipHeightString" select="substring-after($canvasString, 'x')" />

                <xsl:variable name="topBoundary" select="number($clipHeightString) div 3" />
                <xsl:variable name="centreYBoundary" select="2 * number($clipHeightString) div 3" />

                <xsl:variable name="leftBoundary" select="number($clipWidthString) div 3" />
                <xsl:variable name="centreXBoundary" select="2 * number($clipWidthString) div 3" />

                <xsl:variable name="xLocationStr">
                    <xsl:choose>
                        <xsl:when test="number($regionCentreX) &lt; number($leftBoundary)">
                            Left
                        </xsl:when>
                        <xsl:when test="number($regionCentreX) &lt; number($centreXBoundary)">
                            Centre
                        </xsl:when>
                        <xsl:otherwise>
                            Right
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>

                <xsl:variable name="yLocationStr">
                    <xsl:choose>
                        <xsl:when test="number($regionCentreY) &lt; number($topBoundary)">
                            Top
                        </xsl:when>
                        <xsl:when test="number($regionCentreY) &lt; number($centreYBoundary)">
                            Centre
                        </xsl:when>
                        <xsl:otherwise>
                            Bottom
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>

                <xsl:choose>
                    <xsl:when test="$xLocationStr = $yLocationStr">
                        Centre
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="concat($yLocationStr,'-', $xLocationStr)"/>
                    </xsl:otherwise>
                </xsl:choose>

            </xsl:when>
            <xsl:otherwise>
                <xsl:text>All</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- recursive function to count up total duration of shots -->
    <xsl:template name="executiveSummary">
        <xsl:param name="currSumSecs" />
        <xsl:param name="currSumSize" />
        <xsl:param name="count" />
        <xsl:param name="runningProd" />
        <xsl:param name="runningDate" />
        <xsl:param name="anyExpIndex" />
        <xsl:param name="anyWhiteBal" />
        <xsl:param name="anyTint" />

        <xsl:variable name="numShotFrames" select="CodexShotXMLList/CodexShotMetadata[number($count)]/Timecode/Channel/Samples" />
        <xsl:variable name="ourFrameRateEnum" select="CodexShotXMLList/CodexShotMetadata[number($count)]/Video/VideoFormat/FrameRate" />
        <xsl:variable name="frameRate">
            <xsl:call-template name="frameRateDecode">
                <xsl:with-param name="frameRateEnum">
                    <xsl:value-of select="$ourFrameRateEnum"/>
                </xsl:with-param>
            </xsl:call-template>
        </xsl:variable>

        <xsl:variable name="shotDuration" select="$numShotFrames div $frameRate" />
        <xsl:variable name="sumSecs" select="number($currSumSecs) + number($shotDuration)" />

        <xsl:variable name="shotSize" select="CodexShotXMLList/CodexShotMetadata[number($count)]/SizeInGB" />
        <xsl:variable name="sumSize" select="number($currSumSize) + number($shotSize)" />

        <xsl:variable name="curProd">
            <xsl:choose>
                <xsl:when test="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ProductionName != ''">
                    <xsl:value-of select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ProductionName" />
                </xsl:when>
            <xsl:otherwise>&lt;Not Set&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="prod">
            <xsl:choose>
                <xsl:when test="$runningProd = 0 or $runningProd = $curProd">
                    <xsl:value-of select="$curProd" />
                </xsl:when>
                <xsl:otherwise>&lt;mixed&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curDate">
            <xsl:choose>
                <xsl:when test="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ShootDate != ''">
                    <xsl:value-of select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ShootDate" />
                </xsl:when>
                <xsl:otherwise>&lt;Not Set&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="date">
            <xsl:choose>
                <xsl:when test="$runningDate = 0 or $runningDate = $curDate">
                    <xsl:value-of select="$curDate" />
                </xsl:when>
                <xsl:otherwise>&lt;mixed&gt;</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curExpIndex" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/ExposureIndex" />
        <xsl:variable name="expIndexFlag">
            <xsl:choose>
                <xsl:when test="$anyExpIndex = 1 or $curExpIndex != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curWhiteBal" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/WhiteBalance" />
        <xsl:variable name="whiteBalFlag">
            <xsl:choose>
                <xsl:when test="$anyWhiteBal = 1 or $curWhiteBal != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:variable name="curTint" select="CodexShotXMLList/CodexShotMetadata[number($count)]/PropertyStrings/GreenTint" />
        <xsl:variable name="tintFlag">
            <xsl:choose>
                <xsl:when test="$anyTint = 1 or $curTint != ''">1</xsl:when>
                <xsl:otherwise>0</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:choose>
            <xsl:when test="number($count) &gt; 1">
                <xsl:call-template name="executiveSummary">
                    <xsl:with-param name="currSumSecs"><xsl:value-of select="$sumSecs" /></xsl:with-param>
                    <xsl:with-param name="currSumSize"><xsl:value-of select="$sumSize" /></xsl:with-param>
                    <xsl:with-param name="count"><xsl:value-of select="number($count - 1)" /></xsl:with-param>
                    <xsl:with-param name="runningProd"><xsl:value-of select="$prod" /></xsl:with-param>
                    <xsl:with-param name="runningDate"><xsl:value-of select="$date" /></xsl:with-param>
                    <xsl:with-param name="anyExpIndex"><xsl:value-of select="$expIndexFlag" /></xsl:with-param>
                    <xsl:with-param name="anyWhiteBal"><xsl:value-of select="$whiteBalFlag" /></xsl:with-param>
                    <xsl:with-param name="anyTint"><xsl:value-of select="$tintFlag" /></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <results>
                    <duration><xsl:value-of select="$sumSecs" /></duration>
                    <size><xsl:value-of select="$sumSize" /></size>
                    <productionName><xsl:value-of select="$prod" /></productionName>
                    <shootDate><xsl:value-of select="$date" /></shootDate>
                    <exposureIndexSet><xsl:value-of select="$expIndexFlag" /></exposureIndexSet>
                    <whiteBalanceSet><xsl:value-of select="$whiteBalFlag" /></whiteBalanceSet>
                    <greenTintSet><xsl:value-of select="$tintFlag" /></greenTintSet>
                </results>
            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>

    <!-- the main gubbins -->
    <xsl:template match="/">

        <html>

            <head>
              <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
              <link rel="stylesheet" type="text/css" href="report.css" />
              <title>Codex QC Report</title>
            </head>

            <body>

                <!-- set-up some (possibly) useful variables -->

                <xsl:variable name="numShots" select="count(CodexShotXMLList/CodexShotMetadata)" />
                <xsl:variable name="totalFrames" select="sum(CodexShotXMLList/CodexShotMetadata/Video/Channel/Samples)" />

                <xsl:variable name="temp">
                    <xsl:call-template name="executiveSummary">
                        <xsl:with-param name="currSumSecs">0</xsl:with-param>
                        <xsl:with-param name="currSumSize">0</xsl:with-param>
                        <xsl:with-param name="count"><xsl:value-of select="$numShots" /></xsl:with-param>
                        <xsl:with-param name="runningProd">0</xsl:with-param>
                        <xsl:with-param name="runningDate">0</xsl:with-param>
                        <xsl:with-param name="anyExpIndex">0</xsl:with-param>
                        <xsl:with-param name="anyWhiteBal">0</xsl:with-param>
                        <xsl:with-param name="anyTint">0</xsl:with-param>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:variable name="totalDurationSecs" select="exsl:node-set($temp)/results/duration" />
                <xsl:variable name="totalSizeGB" select="exsl:node-set($temp)/results/size" />
                <xsl:variable name="productionName" select="exsl:node-set($temp)/results/productionName" />
                <xsl:variable name="shootDate" select="exsl:node-set($temp)/results/shootDate" />

                <xsl:variable name="hours" select="floor($totalDurationSecs div (60*60))" />
                <xsl:variable name="mins" select="floor(($totalDurationSecs div 60) - ($hours*60))" />
                <xsl:variable name="secs" select="ceiling($totalDurationSecs) mod 60" />

                <xsl:variable name="author" select="CodexShotXMLList/Author" />
                <xsl:variable name="overrideEnum" select="CodexShotXMLList/FrameRateOverride" />

                <xsl:variable name="overrideFps">
                    <xsl:call-template name="frameRateDecode">
                        <xsl:with-param name="frameRateEnum"><xsl:value-of select="$overrideEnum"/></xsl:with-param>
                    </xsl:call-template>
                </xsl:variable>

                <!-- Print logo, header and column titles -->
                <table width="1000" height="98" border="0" cellpadding="0" cellspacing="0">

                    <!-- logo -->
                    <tr>
                        <td colspan="18"><div align="right"><img src="qc-header.gif" alt="" width="709" height="111" /></div></td>
                    </tr>
                    <tr>
                        <td colspan="18">
                            <img src="report-transp.gif" alt="" width="14" height="10" />
                        </td>
                    </tr>

                    <!-- production/author/shootdate -->
                    <tr>
                        <td colspan="18"><table width="100%" border="0" cellpadding="0" cellspacing="0" bgcolor="#CCCCCC">
                            <tr>
                                <td width="14" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="14" height="12" />
                                </td>

                                <td width="292" valign="middle" bgcolor="#dfdfdf"><div align="left">
                                    <strong>PRODUCTION: </strong>
                                    <xsl:value-of select="$productionName" />
                                </div></td>

                                <td width="70" valign="middle" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="14" height="12" />
                                </td>

                                <td width="292" valign="middle" bgcolor="#dfdfdf"><div align="left">
                                  <xsl:if test="$author != ''">
                                    <strong>AUTHOR: </strong>
                                    <xsl:value-of select="$author" />
                                    <xsl:text>; </xsl:text>
                                  </xsl:if>
                                  <xsl:value-of select="CodexShotXMLList/ReportDate" />
                                </div></td>

                                <td width="45" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="100" height="20" />
                                </td>

                                <td width="207" valign="middle" bgcolor="#dfdfdf"><div align="right"><strong>SHOOT DATE: </strong>
                                  <xsl:variable name="shootDateTidy">
                                    <xsl:call-template name="formatDate">
                                      <xsl:with-param name="dateString">
                                        <xsl:value-of select="$shootDate"/>
                                      </xsl:with-param>
                                    </xsl:call-template>
                                  </xsl:variable>
                                  <xsl:value-of select="$shootDateTidy" />
                                </div></td>

                                <td width="18" height="20" bgcolor="#dfdfdf">
                                    <img src="report-transp.gif" alt="" width="18" height="20" />
                                </td>
                            </tr>
                        </table></td>
                    </tr>

                    <!-- summary line -->
                    <tr>
                        <td colspan="18"><img src="report-transp.gif" alt="" width="14" height="14" /></td>
                    </tr>

                    <tr>
                        <td colspan="18">
                            <table width="100%" border="0" cellpadding="0" cellspacing="0" bgcolor="#CCCCCC">
                                <tr>
                                    <td width="14" height="20" bgcolor="#dfdfdf">
                                        <img src="report-transp.gif" alt="" width="14" height="20" />
                                    </td>

                                    <td valign="middle" bgcolor="#dfdfdf"><div align="left"><strong>SUMMARY: </strong>

                                        <xsl:value-of select="$numShots" />
                                        <xsl:choose>
                                            <xsl:when test="number($numShots) > 1">
                                                <xsl:text> clips; </xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text> clip; </xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>

                                        <xsl:value-of select="format-number($totalFrames, ',###,###')" />
                                        <xsl:choose>
                                            <xsl:when test="number($totalFrames) > 1">
                                                <xsl:text> frames; </xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text> frame; </xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>

                                        <xsl:choose>
                                            <xsl:when test="string(number($totalSizeGB))='NaN'">0</xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="format-number($totalSizeGB, ',###,###')" />
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:text> GB; </xsl:text>

                                        <xsl:if test="number($hours) > 0">
                                            <xsl:value-of select="$hours" />
                                            <xsl:choose>
                                                <xsl:when test="number($hours)=1">
                                                    <xsl:text> hr </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> hrs </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test="number($mins) > 0">
                                            <xsl:value-of select="$mins" />
                                            <xsl:choose>
                                                <xsl:when test="number($mins)=1">
                                                    <xsl:text> min </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> mins </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test="number($secs) > 0">
                                            <xsl:value-of select="ceiling($secs)" />
                                            <xsl:choose>
                                                <xsl:when test="number($secs)=1">
                                                    <xsl:text> sec </xsl:text>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:text> secs </xsl:text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:if>
                                        <xsl:if test= "(string(number($secs))='NaN') and (string(number($hours))='NaN') and (string(number($secs))='NaN')">
                                            <xsl:text> 0 secs</xsl:text>
                                        </xsl:if>

                                        <xsl:if test="$overrideFps != '???'">
                                            <xsl:text> (project frame rate: </xsl:text>
                                            <xsl:value-of select="$overrideFps"/>
                                            <xsl:text>fps)</xsl:text>
                                        </xsl:if>
                                    </div></td>
                                </tr>
                            </table>
                        </td>
                    </tr>

                    <!-- gap between summary and table column headings-->
                    <tr>
                        <td height="15" colspan="18" align="left" valign="top">
                            <img src="report-transp.gif" alt="" width="14" height="14" />
                        </td>
                    </tr>

                    <!-- table column headings -->
                    <tr>

                        <td width="10" height="17" valign="middle" bgcolor="#dfdfdf"><h5>
                            <img src="report-transp.gif" alt="" width="11" height="20" />
                        </h5></td>

                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="left">ROLL</h5></td>
                        <td height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="left">CLIP</h5></td>
                        <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">SCENE</h5></td>
                        <td width="40" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">TAKE</h5></td>
                        <td width="50" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">TYPE</h5></td>

                        <td widht="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">START TC</h5></td>
                        <td width="70" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">END TC</h5></td>
                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">AFFECTED FRAMES</h5></td>
                        <td width="60" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">DESCRIPTION</h5></td>
                        <td width="30" height="17" valign="middle" bgcolor="#dfdfdf"><h5 align="center">REGION LOCATION</h5></td>

                        <td width="10" height="17" valign="middle" bgcolor="#dfdfdf"><h5>
                            <img src="report-transp.gif" alt="" width="18" height="17" />
                        </h5></td>
                    </tr>

                    <!-- loop round printing details for each shot -->
                    <xsl:for-each select="CodexShotXMLList/CodexShotMetadata">

                        <!-- set sorting up-->
                        <xsl:sort select="PropertyStrings/OriginalRoll" data-type="text" order="ascending" />
                        <xsl:sort select="Timecode/Channel/StartTc" data-type="text" order="ascending" />

                        <xsl:variable name="sortedIdx" select="position()"/>

                        <!-- create some per shot variables -->
                        <xsl:variable name="tcFrameRate" select="Timecode/Channel/TcFormat/Rate"/>
                        <xsl:variable name="tcFieldRate">
                          <xsl:choose>
                            <xsl:when test="Timecode/Channel/TcFormat/Field = 'true'">2</xsl:when>
                            <xsl:otherwise>1</xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>
                        <xsl:variable name="tcDropFlag" select="Timecode/Channel/TcFormat/DropFrame"/>
                        <xsl:variable name="frameRate">
                              <xsl:call-template name="frameRateDecode">
                                  <xsl:with-param name="frameRateEnum">
                                      <xsl:value-of select="Video/VideoFormat/FrameRate"/>
                                  </xsl:with-param>
                              </xsl:call-template>
                        </xsl:variable>

                        <xsl:variable name="numFrames" select="Video/Channel/Samples"/>

                        <xsl:variable name="startTc" select="Timecode/Channel/StartTc"/>
                        <xsl:variable name="tcStartH" select="substring($startTc, 1, 2)"/>
                        <xsl:variable name="tcStartM" select="substring($startTc, 4, 2)"/>
                        <xsl:variable name="tcStartS" select="substring($startTc, 7, 2)"/>
                        <xsl:variable name="tcStartF" select="substring($startTc, 10, 2)"/>

                        <xsl:variable name="projectFps">
                          <xsl:choose>
                            <xsl:when test="$overrideFps != '???'">
                              <xsl:value-of select="$overrideFps" />
                            </xsl:when>
                            <xsl:otherwise><xsl:value-of select="$frameRate" /></xsl:otherwise>
                          </xsl:choose>
                        </xsl:variable>

                        <xsl:variable name="startFrames" select="((((($tcStartH*60)+$tcStartM)*60)+$tcStartS)*ceiling($frameRate))+$tcStartF"/>
                        <xsl:variable name="projTcStartH" select="floor($startFrames div (60*60*ceiling($projectFps)))"/>
                        <xsl:variable name="projTcStartM" select="floor($startFrames div (60*ceiling($projectFps))) - floor($projTcStartH * 60)" />
                        <xsl:variable name="projTcStartS" select="floor($startFrames div ceiling($projectFps)) mod 60" />
                        <xsl:variable name="projTcStartF" select="$startFrames mod ceiling($projectFps)" />

                        <xsl:variable name="clipRoll" select="PropertyStrings/OriginalRoll" />
                        <xsl:variable name="clipName" select="PropertyStrings/ShotName" />
                        <xsl:variable name="clipScene" select="PropertyStrings/Scene" />
                        <xsl:variable name="clipTake" select="PropertyStrings/Take" />


                        <xsl:if test="not(MarkerList/QC)">
                            <!-- print the table row for this shot -->
                            <tr height="12">

                                <!-- set background colour for alternating stripes -->
                                <xsl:if test="(number($sortedIdx) mod 2) = 0">
                                    <xsl:attribute name="bgcolor">#dfdfdf</xsl:attribute>
                                </xsl:if>

                                <td><h6><xsl:text>&#160;</xsl:text></h6></td>

                                <td><h6 align="left"><xsl:value-of select="$clipRoll" /></h6></td>
                                <td><h6 align="left"><xsl:value-of select="$clipName" /></h6></td>
                                <td><h6 align="center"><xsl:value-of select="$clipScene" /></h6></td>
                                <td><h6 align="center"><xsl:value-of select="$clipTake" /></h6></td>

                                <td><h6 align="center">No Issues</h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6 align="center"></h6></td>
                                <td><h6><xsl:text>&#160;</xsl:text></h6></td>
                            </tr>
                        </xsl:if>

                        <!-- For each QC Mark in the shot -->
                        <xsl:for-each select="MarkerList/QC">

                        <!-- Define some variables about the QC Mark -->
                        <xsl:variable name="qcmarkcount" select="position()" />

                        <xsl:variable name="markType"><xsl:call-template name="getQCMarkType" /></xsl:variable>

                        <xsl:variable name="framesInMark">
                            <xsl:call-template name="getFramesInQCMark">
                                <xsl:with-param name="framesInClip"><xsl:value-of select="$numFrames"/></xsl:with-param>
                            </xsl:call-template>
                        </xsl:variable>

                        <xsl:variable name="regionX"><xsl:call-template name="getRegionXOffset" /></xsl:variable>
                        <xsl:variable name="regionY"><xsl:call-template name="getRegionYOffset" /></xsl:variable>
                        <xsl:variable name="regionW"><xsl:call-template name="getRegionWidth" /></xsl:variable>
                        <xsl:variable name="regionH"><xsl:call-template name="getRegionHeight" /></xsl:variable>
                        <xsl:variable name="firstFrameInMark"><xsl:call-template name="getFirstFrameInQCMark" /></xsl:variable>
                        <xsl:variable name="regionLocation"><xsl:call-template name="getRegionLocation" /></xsl:variable>

                        <!-- print the table row for this shot -->
                        <tr height="12">

                            <!-- set background colour for alternating stripes -->
                            <xsl:if test="(number($sortedIdx) mod 2) = 0">
                                <xsl:attribute name="bgcolor">#dfdfdf</xsl:attribute>
                            </xsl:if>

                            <td><h6>
                                <xsl:text>&#160;</xsl:text>
                            </h6></td>

                            <!-- Clip Name and Roll (Shown for the first QC Mark only) -->
                            <td><h6 align="left">
                                <xsl:if test="number($qcmarkcount) = 1" >
                                    <xsl:value-of select="$clipRoll" />
                                </xsl:if>
                            </h6></td>

                            <td><h6 align="left">
                                <xsl:if test="number($qcmarkcount) = 1" >
                                    <xsl:value-of select="$clipName" />
                                </xsl:if>
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:if test="number($qcmarkcount) = 1" >
                                    <xsl:value-of select="$clipScene" />
                                </xsl:if>
                            </h6></td>

                            <td><h6 align="center">
                                <xsl:if test="number($qcmarkcount) = 1" >
                                    <xsl:value-of select="$clipTake" />
                                </xsl:if>
                            </h6></td>

                            <!-- QC Mark Type -->
                            <td><h6 align="center"><xsl:value-of select="$markType" /></h6></td>

                            <!-- Timecode range covering QC Mark -->
                            <td><h6 align="center">
                                <xsl:choose>
                                    <xsl:when test="$startTc != ''">
                                        <xsl:variable name="shotTcRate" select="$tcFrameRate * $tcFieldRate" />
                                        <xsl:variable name="tcStartFrames" select="((((($tcStartH*60)+$tcStartM)*60)+$tcStartS)*ceiling($shotTcRate))+$tcStartF"/>
                                        <xsl:variable name="tcMarkStartFrames" select="$tcStartFrames + $firstFrameInMark" />
                                        <xsl:variable name="markTcStartH" select="floor($tcMarkStartFrames div (60*60*ceiling($shotTcRate)))" />
                                        <xsl:variable name="markTcStartM" select="floor($tcMarkStartFrames div (60*ceiling($shotTcRate))) - floor($markTcStartH * 60)"/>
                                        <xsl:variable name="markTcStartS" select="floor($tcMarkStartFrames div ceiling($shotTcRate)) mod 60" />
                                        <xsl:variable name="markTcStartF" select="$tcMarkStartFrames mod ceiling($shotTcRate)" />

                                        <xsl:value-of select="format-number($markTcStartH, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($markTcStartM, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($markTcStartS, '00')"/>
                                        <xsl:choose>
                                            <xsl:when test="$tcDropFlag = 'true'">
                                                <xsl:text>;</xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text>:</xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:value-of select="format-number($markTcStartF, '00')"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:text>??:??:??:??</xsl:text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h6></td>


                            <td><h6 align="center">
                                <xsl:choose>
                                    <xsl:when test="$startTc != ''">
                                        <xsl:variable name="shotTcRate" select="$tcFrameRate * $tcFieldRate" />
                                        <xsl:variable name="tcStartFrames" select="((((($tcStartH*60)+$tcStartM)*60)+$tcStartS)*ceiling($shotTcRate))+$tcStartF"/>
                                        <xsl:variable name="tcMarkEndFrames" select="$tcStartFrames + $firstFrameInMark + $framesInMark - 1" />
                                        <xsl:variable name="markTcEndH" select="floor($tcMarkEndFrames div (60*60*ceiling($shotTcRate)))" />
                                        <xsl:variable name="markTcEndM" select="floor($tcMarkEndFrames div (60*ceiling($shotTcRate))) - floor($markTcEndH * 60)"/>
                                        <xsl:variable name="markTcEndS" select="floor($tcMarkEndFrames div ceiling($shotTcRate)) mod 60" />
                                        <xsl:variable name="markTcEndF" select="$tcMarkEndFrames mod ceiling($shotTcRate)" />

                                        <xsl:value-of select="format-number($markTcEndH, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($markTcEndM, '00')"/>
                                        <xsl:text>:</xsl:text>
                                        <xsl:value-of select="format-number($markTcEndS, '00')"/>
                                        <xsl:choose>
                                            <xsl:when test="$tcDropFlag = 'true'">
                                                <xsl:text>;</xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text>:</xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:value-of select="format-number($markTcEndF, '00')"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:text>??:??:??:??</xsl:text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h6></td>

                            <!-- Frames affected by the QC Mark -->
                            <td><h6 align="center"><xsl:value-of select="$framesInMark" /></h6></td>

                            <!-- QC Mark Description -->
                            <td><h6 align="center"><xsl:value-of select="Description"/></h6></td>

                            <!-- QC Region Location -->
                            <td><h6 align="center"><xsl:value-of select="$regionLocation" /></h6></td>

                            <td><h6><xsl:text>&#160;</xsl:text></h6></td>
                        </tr>
                    </xsl:for-each>
                    </xsl:for-each>
                </table>
            </body>
        </html>
    </xsl:template>
</xsl:stylesheet>
