#!/bin/bash
###########################################################################
#
# create_xl_xfs_raid
#
# Author: Iain Collins 07/06/16
#
# The purpose of this script is to create a XFS raid volume using the
# given device names. The number of devices is fixed to 6 as this will
# be used to create the XFS volume on a Vault XL which has additional
# SSDs fitted
#
###########################################################################

badParameters() {
    echo "Usage create_xl_xfs_raid.sh <device 1 name> <device 2 name> <device 3 name> <device 4 name> <device 5 name> <device 6 name>"
    echo "Device name should be in the format of /dev/sdX"
    exit -1
}

# Check the number of arguments
if [ $# -ne 6 ]; then
    echo "ERROR: Invalid number of argument provided"
    badParameters
fi

# Display a warning to the user to remove all drives and stop drserver
echo "This script will stop drserver and stop any software raids which may be running."
echo "This will copy the existing contents of /etc/mdadm.conf to /etc/mdadm.conf.bak and create a new mdadm.conf file"
echo "Enter y to continue."
read -r input
if [ "$input" != "Y" ] && [ "$input" != "y" ]; then
        echo "EXITING: Cancelled by user"
        exit 0
fi

# Attempt to stop drserver and any software raids which may still be running
service drserver stop
find /dev -name 'md*' -type b -exec /bin/umount {} \;
find /dev -name 'md*' -type b -exec /usr/bin/stopraid.sh {} \;

# Ensure eash argument is a device name and that it exists
for var in "$@"
do
    driveName="$var"
    if [ ! -e $driveName ]; then
        echo "ERROR: $driveName does not exist"
        badParameters
    fi

    altPartitionName="$driveName"2
    if [ -e $altPartitionName ]; then
        echo "ERROR: $altPartitionName exists - ensure that system disk has not been selected by mistake"
        badParameters
    fi
done

# Issue the are you sure warning
echo "WARNING This script will create a XFS raid using the following drives:"
for var in "$@"
do
    echo "    $var"
done
echo "Are you sure you want to continue (y/n)?"
read input
if [ "$input" != "Y" ] && [ "$input" != "y" ]; then
    echo "EXITING: Cancelled by user"
    exit 0
fi

# Create a new raid
# The sleep and udevadm settle are there to allow enough time for the system
# to pickup and register the newly created partitions
echo "Creating the new raid"
/usr/bin/clear.sh "$1" "$2" "$3" "$4" "$5" "$6" > /dev/null
sleep 3
/sbin/udevadm settle
/usr/bin/raid.sh XXL XXL xfs /dev/md101 5 "$1" "$2" "$3" "$4" "$5" "$6"

# Get the info
raidInfo="$(mdadm --detail --scan)"

# Generate the new /etc/mdadm.conf file
# This uses the last 40 characters of raidInfo "UUID=abcdefg:abcdefg:abcdefg:abcdefg" to define how the new raid will be assembled
if [ -e /etc/mdadm.conf ]; then
    mv /etc/mdadm.conf /etc/mdadm.conf.bak
fi
echo "MAILADDR root" >> /etc/mdadm.conf
echo "ARRAY /dev/md101 ${raidInfo:(-40)}" >> /etc/mdadm.conf
echo "AUTO +imsm +1.x -all" >> /etc/mdadm.conf

echo "Complete"
exit 0
